document.addEventListener("DOMContentLoaded", () => {
  const userSearchInput = document.getElementById("userSearchInput")
  const userSearchButton = document.getElementById("userSearchButton")

  if (userSearchButton) {
    userSearchButton.addEventListener("click", performUserSearch)
  }
  if (userSearchInput) {
    userSearchInput.addEventListener("keypress", (e) => {
      if (e.key === "Enter") {
        performUserSearch()
      }
    })
  }

  loadPendingRequests()
  loadCurrentFriends()
})

function performUserSearch() {
  const query = document.getElementById("userSearchInput").value.trim()
  const resultsContainer = document.getElementById("userSearchResults")
  resultsContainer.innerHTML = '<p class="loading-message">Recherche en cours...</p>'

  if (query.length < 2) {
    resultsContainer.innerHTML = '<p class="info-message">Veuillez entrer au moins 2 caractères.</p>'
    return
  }

  fetch(`api/users_search.php?q=${encodeURIComponent(query)}`)
    .then((response) => response.json())
    .then((result) => {
      if (result.success) {
        renderUserSearchResults(result.data)
      } else {
        resultsContainer.innerHTML = `<p class="error-message">Erreur: ${result.error || "Impossible de rechercher."}</p>`
      }
    })
    .catch((error) => {
      resultsContainer.innerHTML = `<p class="error-message">Erreur de connexion: ${error.message}</p>`
      console.error("Error searching users:", error)
    })
}

function renderUserSearchResults(users) {
  const container = document.getElementById("userSearchResults")
  container.innerHTML = ""

  if (users.length === 0) {
    container.innerHTML = '<p class="info-message">Aucun utilisateur trouvé.</p>'
    return
  }

  users.forEach((user) => {
    const userDiv = document.createElement("div")
    userDiv.className = "user-search-item friend-item" // Reuse friend-item style
    const avatarUrl = user.avatar_url || "assets/images/avatar-placeholder.jpg"

    let actionButtonHTML = ""
    if (user.friend_status === "friends") {
      actionButtonHTML = '<button class="btn-friend-action disabled" disabled>Amis</button>'
    } else if (user.friend_status === "request_sent") {
      actionButtonHTML = '<button class="btn-friend-action disabled" disabled>Demande envoyée</button>'
    } else if (user.friend_status === "request_received") {
      actionButtonHTML = `
                <button class="btn-friend-action btn-accept" data-user-id="${user.id}">Accepter</button>
                <button class="btn-friend-action btn-decline" data-user-id="${user.id}">Refuser</button>
            `
    } else {
      // Not friends, no pending request
      actionButtonHTML = `<button class="btn-friend-action btn-add-friend" data-user-id="${user.id}">Ajouter</button>`
    }

    userDiv.innerHTML = `
            <img src="${escapeHTML(avatarUrl)}" alt="Avatar de ${escapeHTML(user.display_name)}" class="friend-avatar">
            <span class="friend-name">${escapeHTML(user.display_name)}</span>
            <div class="friend-actions">
                ${actionButtonHTML}
            </div>
        `
    container.appendChild(userDiv)
  })
  attachActionListenersToSearchResults()
}

function attachActionListenersToSearchResults() {
  document.querySelectorAll("#userSearchResults .btn-add-friend").forEach((button) => {
    button.addEventListener("click", (e) => handleFriendAction("send_request", e.target.dataset.userId, e.target))
  })
  document.querySelectorAll("#userSearchResults .btn-accept").forEach((button) => {
    button.addEventListener("click", (e) =>
      handleFriendAction("accept_request", e.target.dataset.userId, e.target.closest(".user-search-item")),
    )
  })
  document.querySelectorAll("#userSearchResults .btn-decline").forEach((button) => {
    button.addEventListener("click", (e) =>
      handleFriendAction("decline_request", e.target.dataset.userId, e.target.closest(".user-search-item")),
    )
  })
}

function loadPendingRequests() {
  const container = document.getElementById("pendingRequestsList")
  container.innerHTML = '<p class="loading-message">Chargement des demandes...</p>'
  fetch("api/friends.php?action=list_pending_requests")
    .then((response) => response.json())
    .then((result) => {
      if (result.success) {
        renderPendingRequests(result.data)
      } else {
        container.innerHTML = `<p class="error-message">Erreur: ${result.error || "Impossible de charger les demandes."}</p>`
      }
    })
    .catch((error) => {
      container.innerHTML = `<p class="error-message">Erreur de connexion: ${error.message}</p>`
      console.error("Error fetching pending requests:", error)
    })
}

function renderPendingRequests(requests) {
  const container = document.getElementById("pendingRequestsList")
  container.innerHTML = ""

  if (requests.length === 0) {
    container.innerHTML = '<p class="info-message">Aucune demande d\'ami en attente.</p>'
    return
  }

  requests.forEach((request) => {
    const requestDiv = document.createElement("div")
    requestDiv.className = "friend-item pending-request-item"
    const avatarUrl = request.avatar_url || "assets/images/avatar-placeholder.jpg"
    requestDiv.innerHTML = `
            <img src="${escapeHTML(avatarUrl)}" alt="Avatar de ${escapeHTML(request.display_name)}" class="friend-avatar">
            <span class="friend-name">${escapeHTML(request.display_name)}</span>
            <div class="friend-actions">
                <button class="btn-friend-action btn-accept" data-user-id="${request.friend_user_id}">Accepter</button>
                <button class="btn-friend-action btn-decline" data-user-id="${request.friend_user_id}">Refuser</button>
            </div>
        `
    container.appendChild(requestDiv)
  })

  document.querySelectorAll("#pendingRequestsList .btn-accept").forEach((button) => {
    button.addEventListener("click", (e) =>
      handleFriendAction("accept_request", e.target.dataset.userId, e.target.closest(".pending-request-item")),
    )
  })
  document.querySelectorAll("#pendingRequestsList .btn-decline").forEach((button) => {
    button.addEventListener("click", (e) =>
      handleFriendAction("decline_request", e.target.dataset.userId, e.target.closest(".pending-request-item")),
    )
  })
}

function loadCurrentFriends() {
  const container = document.getElementById("currentFriendsList")
  container.innerHTML = '<p class="loading-message">Chargement des amis...</p>'
  fetch("api/friends.php?action=list_friends")
    .then((response) => response.json())
    .then((result) => {
      if (result.success) {
        renderCurrentFriends(result.data)
      } else {
        container.innerHTML = `<p class="error-message">Erreur: ${result.error || "Impossible de charger la liste d'amis."}</p>`
      }
    })
    .catch((error) => {
      container.innerHTML = `<p class="error-message">Erreur de connexion: ${error.message}</p>`
      console.error("Error fetching current friends:", error)
    })
}

function renderCurrentFriends(friends) {
  const container = document.getElementById("currentFriendsList")
  container.innerHTML = ""

  if (friends.length === 0) {
    container.innerHTML = '<p class="info-message">Vous n\'avez aucun ami pour le moment.</p>'
    return
  }

  friends.forEach((friend) => {
    const friendDiv = document.createElement("div")
    friendDiv.className = "friend-item"
    const avatarUrl = friend.avatar_url || "assets/images/avatar-placeholder.jpg"
    friendDiv.innerHTML = `
            <img src="${escapeHTML(avatarUrl)}" alt="Avatar de ${escapeHTML(friend.display_name)}" class="friend-avatar">
            <span class="friend-name">${escapeHTML(friend.display_name)}</span>
            <div class="friend-actions">
                <button class="btn-friend-action btn-more-options" data-user-id="${friend.friend_user_id}">⋮</button>
                <div class="friend-options-menu" style="display:none;">
                    <button class="btn-friend-action btn-remove-friend" data-user-id="${friend.friend_user_id}">Retirer</button>
                </div>
            </div>
        `
    container.appendChild(friendDiv)
  })

  document.querySelectorAll("#currentFriendsList .btn-more-options").forEach((button) => {
    button.addEventListener("click", (e) => {
      const menu = e.target.nextElementSibling
      menu.style.display = menu.style.display === "none" ? "block" : "none"
    })
  })
  document.querySelectorAll("#currentFriendsList .btn-remove-friend").forEach((button) => {
    button.addEventListener("click", (e) => {
      if (
        confirm(
          `Êtes-vous sûr de vouloir retirer ${e.target.closest(".friend-item").querySelector(".friend-name").textContent} de vos amis ?`,
        )
      ) {
        handleFriendAction("remove_friend", e.target.dataset.userId, e.target.closest(".friend-item"))
      }
    })
  })
}

function handleFriendAction(action, targetUserId, elementToUpdateOrRemove = null) {
  const formData = new FormData()
  formData.append("action", action)
  formData.append("target_user_id", targetUserId)

  fetch("api/friends.php", {
    method: "POST",
    body: formData,
  })
    .then((response) => response.json())
    .then((result) => {
      if (result.success) {
        showGlobalNotification(result.message, "success")
        // Update UI based on action
        if (action === "send_request" && elementToUpdateOrRemove) {
          elementToUpdateOrRemove.textContent = "Demande envoyée"
          elementToUpdateOrRemove.disabled = true
          elementToUpdateOrRemove.classList.remove("btn-add-friend")
        } else if (
          (action === "accept_request" || action === "decline_request" || action === "remove_friend") &&
          elementToUpdateOrRemove
        ) {
          elementToUpdateOrRemove.remove()
        }
        // Reload relevant lists
        if (action === "accept_request" || action === "decline_request") {
          loadPendingRequests() // Refresh pending list
          loadCurrentFriends() // Refresh friends list if accepted
        }
        if (action === "remove_friend") {
          loadCurrentFriends() // Refresh friends list
        }
        if (action === "send_request") {
          // Potentially refresh search results if status needs to update there, or just disable button
          // For simplicity, the button text changes. A full refresh of search might be too much.
        }
      } else {
        showGlobalNotification(result.error || "Une erreur est survenue.", "error")
      }
    })
    .catch((error) => {
      console.error(`Error performing friend action ${action}:`, error)
      showGlobalNotification("Erreur de connexion.", "error")
    })
}

function escapeHTML(str) {
  if (str === null || str === undefined) return ""
  const div = document.createElement("div")
  div.appendChild(document.createTextNode(str))
  return div.innerHTML
}

// Assurez-vous que showGlobalNotification est disponible (par exemple, depuis main.js)
function showGlobalNotification(message, type = "info") {
  if (typeof showNotification === "function") {
    showNotification(message, type)
  } else {
    alert(`${type.toUpperCase()}: ${message}`)
  }
}
